<?php
/**
 * Elementor Widget
 * @package Senatory
 * @since 1.0.0
 */

namespace Elementor;
class Senatory_Event_Campaign_One_Widget extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'theme-event-campaign-one-widget';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Event: Campaign 01', 'senatory-core');
    }


    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'bl_icon eicon-archive-title';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['senatory_widgets'];
    }

    /**
     * Register Elementor widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {

        $this->start_controls_section(
            'settings_section',
            [
                'label' => esc_html__('General Settings', 'senatory-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );


        $this->add_control(
            'bg_image',
            [
                'label' => esc_html__('Box Background Image', 'senatory-core'),
                'type' => Controls_Manager::MEDIA,
                'show_label' => false,
                'description' => esc_html__('Upload bg image', 'senatory-core'),
                'default' => [
                    'src' => Utils::get_placeholder_image_src()
                ],
            ]
        );

        $this->add_control('countdown_date', [
            'label' => esc_html__('Countdown Time', 'senatory-core'),
            'type' => Controls_Manager::DATE_TIME,
            'default' => esc_html__('14', 'senatory-core'),
            'description' => esc_html__('enter counterup date', 'senatory-core')
        ]);
        $this->add_control(
            'time_format',
            [
                'label' => esc_html__('Need 24 Hour Time Formate?', 'events-addon-for-elementor'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'events-addon-for-elementor'),
                'label_off' => esc_html__('Hide', 'events-addon-for-elementor'),
                'return_value' => 'true',
            ]
        );


        $this->add_control('total', [
            'label' => esc_html__('Total Posts', 'senatory-core'),
            'type' => Controls_Manager::TEXT,
            'default' => '-1',
            'description' => esc_html__('enter how many post you want in masonry , enter -1 for unlimited post.')
        ]);


        $this->add_control('category', [
            'label' => esc_html__('Category', 'senatory-core'),
            'type' => Controls_Manager::SELECT2,
            'multiple' => false,
            'options' => senatory_core()->get_terms_names('tribe_events_cat'),
            'description' => esc_html__('select category as you want, leave it blank for all category', 'senatory-core'),
        ]);

        $this->add_control('order', [
            'label' => esc_html__('Order', 'senatory-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'ASC' => esc_html__('Ascending', 'senatory-core'),
                'DESC' => esc_html__('Descending', 'senatory-core'),
            ),
            'default' => 'ASC',
            'description' => esc_html__('select order', 'senatory-core')
        ]);


        $this->add_control('orderby', [
            'label' => esc_html__('OrderBy', 'senatory-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                'ID' => esc_html__('ID', 'senatory-core'),
                'title' => esc_html__('Title', 'senatory-core'),
                'date' => esc_html__('Date', 'senatory-core'),
                'rand' => esc_html__('Random', 'senatory-core'),
                'comment_count' => esc_html__('Most Comments', 'senatory-core'),
            ),
            'default' => 'ID',
            'description' => esc_html__('select order', 'senatory-core')
        ]);


        $this->add_control('excerpt_length', [
            'label' => esc_html__('Excerpt Length', 'senatory-core'),
            'type' => Controls_Manager::SELECT,
            'options' => array(
                20 => esc_html__('Short', 'senatory-core'),
                55 => esc_html__('Regular', 'senatory-core'),
                100 => esc_html__('Long', 'senatory-core'),
            ),
            'default' => 15,
            'description' => esc_html__('select excerpt length', 'senatory-core')
        ]);


        $this->add_control(
            'btn_text',
            [
                'label' => esc_html__('Button Text', 'senatory-core'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Discover More', 'senatory-core'),
                'description' => esc_html__('Enter button text', 'senatory-core'),
            ]
        );


        $this->add_control(
            'btn_link',
            [
                'label' => esc_html__('Button URL', 'senatory-core'),
                'type' => Controls_Manager::URL,
                'default' => [
                    'url' => '#'
                ],
                'description' => esc_html__('Enter button url', 'senatory-core'),
            ]
        );


        $this->end_controls_section();

        $this->start_controls_section(
            'styling_settings_section',
            [
                'label' => esc_html__('Styling Settings', 'senatory-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );


        $this->add_control('title_color', [
            'label' => esc_html__('Title Color', 'senatory-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .campaign-list-item .list-single-items .content .title" => "color: {{VALUE}}"
            ]
        ]);

        $this->add_control('title_hover_color', [
            'label' => esc_html__('Title Hover Color', 'senatory-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .campaign-list-item .list-single-items .content .title:hover" => "color: {{VALUE}}"
            ]
        ]);
        $this->add_control('cat_bg_color', [
            'label' => esc_html__('Category Background', 'senatory-core'),
            'type' => Controls_Manager::COLOR,
            'default'   => '#DD131A',
            'selectors' => [
                "{{WRAPPER}} .campaign-list-item .list-single-items .content .designation .event" => "background: {{VALUE}}"
            ]
        ]);
        $this->add_control('description_color', [
            'label' => esc_html__('Description Color', 'senatory-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .campaign-list-item .list-single-items .content p" => "color: {{VALUE}}"
            ]
        ]);

        $this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'border',
				'selector' => '{{WRAPPER}} .campaign-list-item .list-single-items',
			]
		);
        $this->add_control(
			'margin',
			[
				'label' => esc_html__( 'Padding', 'senatory-core' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .campaign-single-item .campaign-list-item' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
        $this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'item_box_shadow',
				'label' => esc_html__( 'Box Shadow', 'senatory-core' ),
				'selector' => '{{WRAPPER}} .campaign-single-item .campaign-list-item',
			]
		);
        $this->add_control('event_cat_styling_divider', [
            'type' => Controls_Manager::DIVIDER
        ]);
        $this->add_control('cat_color', [
            'label' => esc_html__('Category Color', 'senatory-core'),
            'type' => Controls_Manager::COLOR,
            'default'   => '#FFFFFF',
            'selectors' => [
                "{{WRAPPER}} .campaign-list-item .list-single-items .content .designation .event" => "color: {{VALUE}}"
            ]
        ]);

        $this->add_control(
			'cat_bdr_radius',
			[
				'label' => esc_html__( 'Border Radius', 'senatory-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 5,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors' => [
					'{{WRAPPER}} .campaign-list-item .list-single-items .content .designation .event' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);
        $this->add_control('title_styling_divider', [
            'type' => Controls_Manager::DIVIDER
        ]);

        $this->end_controls_section();

        $this->start_controls_section(
            'typography_settings_section',
            [
                'label' => esc_html__('Typography Settings', 'senatory-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Title Typography', 'senatory-core'),
            'name' => 'title_typography',
            'description' => esc_html__('select title typography', 'senatory-core'),
            'selector' => "{{WRAPPER}} .campaign-list-item .list-single-items .content .title"
        ]);


        $this->add_group_control(Group_Control_Typography::get_type(), [
            'label' => esc_html__('Description Typography', 'senatory-core'),
            'name' => 'description_typography',
            'description' => esc_html__('select description typography', 'senatory-core'),
            'selector' => "{{WRAPPER}} .campaign-list-item .list-single-items .content p"
        ]);


        $this->end_controls_section();
    }

    /**
     * Render Elementor widget output on the frontend.
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();

        //query settings
        $total_posts = $settings['total'];
        $category = $settings['category'];
        $order = $settings['order'];
        $orderby = $settings['orderby'];
        global $post;
        $all_events = tribe_get_events(array(
            'posts_per_page' => $total_posts,
            'tribe_events_cat' => $category,
            'orderby' => $orderby,
            'order' => $order,
        ));

        ?>

        <div class="campaign-area-custom">
            <div class="campaign-single-item">
                <div class="campaign-list-item">
                    <?php
                    foreach ($all_events as $data):
                        $post_id = $data->ID;
                        $event_start_time = get_post_meta($post_id, '_EventStartDate', true);
                        $post_categories = get_the_terms($post_id, 'tribe_events_cat');
                        ?>
                        <div class="list-single-items">
                            <div class="content">
                                <div class="designation">
                                    <?php if (!empty($post_categories)): ?>
                                        <span class="event">
                                                    <?php foreach ($post_categories as $cats) : ?>
                                                        <?php echo esc_attr($cats->name) ?>
                                                    <?php endforeach; ?>
                                                </span>
                                    <?php endif; ?>
                                    <span class="date"><?php echo esc_html(date('F j, Y', strtotime($event_start_time))) ?></span>
                                </div>
                                <h5 class="title"><a
                                            href="<?php the_permalink($post_id); ?>"><?php echo esc_html($data->post_title); ?></a>
                                </h5>
                                <p><?php echo senatory_core()->tribe_event_excerpt($settings['excerpt_length'], $data->post_content); ?></p>
                            </div>
                        </div>
                    <?php
                    endforeach;
                    wp_reset_query();
                    ?>
                </div>
            </div>
        </div>


        <?php
    }
}

Plugin::instance()->widgets_manager->register_widget_type(new Senatory_Event_Campaign_One_Widget());