<?php
/**
 * Elementor Widget
 * @package Senatory
 * @since 1.0.0
 */

namespace Elementor;
class Senatory_Image_Gallery_Widget extends Widget_Base
{

    /**
     * Get widget name.
     *
     * Retrieve Elementor widget name.
     *
     * @return string Widget name.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_name()
    {
        return 'theme-image-gallery-widget';
    }

    /**
     * Get widget title.
     *
     * Retrieve Elementor widget title.
     *
     * @return string Widget title.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_title()
    {
        return esc_html__('Image Gallery: 01', 'senatory-core');
    }

    /**
     * Get widget icon.
     *
     * Retrieve Elementor widget icon.
     *
     * @return string Widget icon.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_icon()
    {
        return 'bl_icon eicon-archive-title';
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the Elementor widget belongs to.
     *
     * @return array Widget categories.
     * @since 1.0.0
     * @access public
     *
     */
    public function get_categories()
    {
        return ['senatory_widgets'];
    }

    /**
     * Register Elementor widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {

        $this->start_controls_section(
            'settings_section',
            [
                'label' => esc_html__('General Settings', 'senatory-core'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'image',
            [
                'label' => esc_html__('Image', 'senatory-core'),
                'type' => Controls_Manager::MEDIA,
                'description' => esc_html__('enter image.', 'senatory-core'),
                'default' => array(
                    'url' => Utils::get_placeholder_image_src()
                )
            ]
        );

        $this->add_control(
            'popup_image',
            [
                'label' => esc_html__('Popup Image', 'senatory-core'),
                'type' => Controls_Manager::MEDIA,
                'description' => esc_html__('enter image.', 'senatory-core'),
                'default' => array(
                    'url' => Utils::get_placeholder_image_src()
                )
            ]
        );


        $this->add_control(
            'title',
            [
                'label' => esc_html__('Title', 'senatory-core'),
                'type' => Controls_Manager::TEXT,
                'description' => esc_html__('enter title', 'senatory-core'),
                'default' => esc_html__('Senior Party 2019', 'senatory-core')
            ]
        );

        $this->add_control(
            'img_caption',
            [
                'label' => esc_html__('Caption Show/Hide', 'senatory-core'),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'no',
                'description' => esc_html__('Show/hide caption', 'senatory-core')
            ]
        );


        $this->add_responsive_control(
            'grid_img_height',
            [
                'label' => esc_html__('Set Gallery Image Height', 'evior-extra'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                    ],
                ],
                'devices' => ['desktop', 'tablet', 'mobile'],
                'desktop_default' => [
                    'size' => 110,
                    'unit' => 'px',
                ],
                'tablet_default' => [
                    'size' => 110,
                    'unit' => 'px',
                ],
                'mobile_default' => [
                    'size' => 110,
                    'unit' => 'px',
                ],

                'default' => [
                    'unit' => 'px',
                    'size' => 110,
                ],

                'selectors' => [
                    '{{WRAPPER}} .gallery-single-items .thumb img' => 'height: {{SIZE}}{{UNIT}}!important;',
                ],
            ]
        );


        $this->end_controls_section();

        $this->start_controls_section(
            'galley_styling_settings_section',
            [
                'label' => esc_html__('Styling Settings', 'senatory-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );


        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'hover_background',
                'label' => esc_html__('Hover Image Overlay', 'senatory-core'),
                'types' => ['classic', 'gradient', 'video'],
                'selector' => '{{WRAPPER}} .gallery-single-items .thumb::before',
            ]
        );


        $this->add_control('title_color', [
            'label' => esc_html__('TItle Color', 'senatory-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .single-image-gallery .gallery-overlay .cart-icon a .title" => "color: {{VALUE}}"
            ]
        ]);

        $this->add_control('icon_color', [
            'label' => esc_html__('Designation Color', 'senatory-core'),
            'type' => Controls_Manager::COLOR,
            'selectors' => [
                "{{WRAPPER}} .single-image-gallery .gallery-overlay .cart-icon a" => "color: {{VALUE}}"
            ]
        ]);

        $this->end_controls_tab();


        $this->add_control('team_typography_divider', [
            'type' => Controls_Manager::DIVIDER
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'name' => 'name_typography',
            'label' => esc_html__('Title Typography', 'senatory-core'),
            'selector' => "{{WRAPPER}} .single-image-gallery .gallery-overlay .cart-icon a .title"
        ]);

        $this->end_controls_section();

    }

    /**
     * Render Elementor widget output on the frontend.
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();

        ?>
        <?php
        $image_id = $settings['image']['id'];
        $image_url = !empty($image_id) ? wp_get_attachment_image_src($image_id, 'full', false)[0] : '';
        $image_alt = !empty($img_id) ? get_post_meta($img_id, '_wp_attachment_image_alt', true) : '';

        $popup_image_id = $settings['popup_image']['id'];
        $popup_image_url = !empty($popup_image_id) ? wp_get_attachment_image_src($popup_image_id, 'full', false)[0] : '';
        ?>

        <div class="gallery-single-items wow animate__animated animate__fadeInLeft">
            <div class="thumb">
                <img src="<?php echo esc_url($image_url); ?>" alt="<?php echo esc_attr($image_alt); ?>">
                <div class="cart-icon">
                    <a class="image-popup" href="<?php echo esc_url($popup_image_url); ?>"><i
                                class="fas fa-plus"></i></a>
                </div>
            </div>
            <?php if ($settings['img_caption'] == 'yes'): ?>
                <div class="content">
                    <h6 class="title"><?php echo esc_html($settings['title']); ?></h6>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }
}

Plugin::instance()->widgets_manager->register_widget_type(new Senatory_Image_Gallery_Widget());