<?php
/**
 * Theme Init Functions
 * @package senatory
 * @since 1.0.0
 */

if (!defined("ABSPATH")) {
    exit(); //exit if access directly
}

if (!class_exists('Senatory_Init')) {

    class Senatory_Init
    {
        /**
         * $instance
         * @since 1.0.0
         */
        protected static $instance;

        public function __construct()
        {
            /*
             * theme setup
             */
            add_action('after_setup_theme', array($this, 'theme_setup'));
            /**
             * Widget Init
             */
            add_action('widgets_init', array($this, 'theme_widgets_init'));
            /**
             * Theme Assets
             */
            add_action('wp_enqueue_scripts', array($this, 'theme_assets'));
            /**
             * Registers an editor stylesheet for the theme.
             */
            add_action('admin_init', array($this, 'add_editor_styles'));
            /**
             * enqueue an editor stylesheet for the gutenbarg.
             */
            add_action('enqueue_block_editor_assets', array($this, 'gutenbarg_script'));
        }

        /**
         * getInstance()
         */
        public static function getInstance()
        {
            if (null == self::$instance) {
                self::$instance = new self();
            }
            return self::$instance;
        }

        /**
         * Theme Setup
         * @since 1.0.0
         */
        public function theme_setup()
        {
            /*
             * Make theme available for translation.
             * Translations can be filed in the /languages/ directory.
             */
            load_theme_textdomain('senatory', get_template_directory() . '/languages');

            // Add default posts and comments RSS feed links to head.
            add_theme_support('automatic-feed-links');

            /*
             * Let WordPress manage the document title.
             */
            add_theme_support('title-tag');

            /*
             * Enable support for Post Thumbnails on posts and pages.
             *
             * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
             */
            add_theme_support('post-thumbnails');

            // This theme uses wp_nav_menu() in one location.
            register_nav_menus(array(
                'main-menu' => esc_html__('Primary Menu', 'senatory'),
            ));

            register_nav_menus(array(
                'center-menu' => esc_html__('Header Center Menu', 'senatory'),
            ));

            /*
             * Switch default core markup for search form, comment form, and comments
             * to output valid HTML5.
             */
            add_theme_support('html5', array(
                'search-form',
                'comment-form',
                'comment-list',
                'gallery',
                'caption',
            ));

            // Add theme support for selective wp block styles
            add_theme_support("wp-block-styles");
            // Add theme support for selective align wide
            add_theme_support("align-wide");
            // Add theme support for selective responsive embeds
            add_theme_support("responsive-embeds");

            // Add theme support for selective refresh for widgets.
            add_theme_support('customize-selective-refresh-widgets');

            /**
             * Add support for core custom logo.
             *
             * @link https://codex.wordpress.org/Theme_Logo
             */
            add_theme_support('custom-logo', array(
                'height' => 250,
                'width' => 250,
                'flex-width' => true,
                'flex-height' => true,
            ));

            //woocommerce support
            add_theme_support('woocommerce');
            add_theme_support('wc-product-gallery-zoom');
            add_theme_support('wc-product-gallery-lightbox');
            add_theme_support('wc-product-gallery-slider');

            //add theme support for post format
            add_theme_support('post-formats', array('image', 'video', 'gallery', 'link', 'quote'));

            // This variable is intended to be overruled from themes.
            $GLOBALS['content_width'] = apply_filters('senatory_content_width', 740);

            //add image sizes
            add_image_size('senatory_classic', 750, 400, true);
            add_image_size('senatory_grid', 370, 270, true);
            add_image_size('senatory_medium', 550, 380, true);
            add_image_size('senatory_give_thumb', 540, 570, true);

            self::load_theme_dependency_files();
        }

        /**
         * Theme Widget Init
         * @since 1.0.0
         * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
         */
        public function theme_widgets_init()
        {
            register_sidebar(array(
                'name' => esc_html__('Sidebar', 'senatory'),
                'id' => 'sidebar-1',
                'description' => esc_html__('Add widgets here.', 'senatory'),
                'before_widget' => '<div id="%1$s" class="widget %2$s">',
                'after_widget' => '</div>',
                'before_title' => '<h4 class="widget-headline style-01">',
                'after_title' => '</h4>',
            ));

            register_sidebar(array(
                'name' => esc_html__('Footer Widget Area', 'senatory'),
                'id' => 'footer-widget',
                'description' => esc_html__('Add widgets here.', 'senatory'),
                'before_widget' => '<div class="col-lg-3 col-md-6"><div id="%1$s" class="widget footer-widget %2$s">',
                'after_widget' => '</div></div>',
                'before_title' => '<h4 class="widget-headline">',
                'after_title' => '</h4>',
            ));

            register_sidebar(array(
                'name' => esc_html__('Footer Widget Area Two', 'senatory'),
                'id' => 'footer-widget-two',
                'description' => esc_html__('Add widgets here.', 'senatory'),
                'before_widget' => '<div class="col-lg-3 col-md-6"><div id="%1$s" class="widget footer-widget %2$s">',
                'after_widget' => '</div></div>',
                'before_title' => '<h4 class="widget-headline">',
                'after_title' => '</h4>',
            ));

            register_sidebar(array(
                'name' => esc_html__('Footer Widget Area Three', 'senatory'),
                'id' => 'footer-widget-three',
                'description' => esc_html__('Add widgets here.', 'senatory'),
                'before_widget' => '<div class="col-lg-3 col-md-6"><div id="%1$s" class="widget footer-widget %2$s">',
                'after_widget' => '</div></div>',
                'before_title' => '<h4 class="widget-headline">',
                'after_title' => '</h4>',
            ));
            register_sidebar(array(
                'name' => esc_html__('Footer Widget Area Five', 'senatory'),
                'id' => 'footer-widget-five',
                'description' => esc_html__('Add widgets here.', 'senatory'),
                'before_widget' => '<div class="col-lg-3 col-md-6 col-sm-6 col-6"><div id="%1$s" class="footer-widget widget widget_nav_menu %2$s">',
                'after_widget' => '</div></div>',
                'before_title' => '<h4 class="widget-title">',
                'after_title' => '</h4>',
            ));


        }

        /**
         * Theme Assets
         * @since 1.0.0
         */
        public function theme_assets()
        {
            self::load_theme_css();
            self::load_theme_js();
        }

        /*
         * Load theme options google fonts css
         * @since 1.0.0
         */
        public static function load_google_fonts()
        {

            $enqueue_fonts = [];
            //body font enqueue
            $body_font = cs_get_option('_body_font') ? cs_get_option('_body_font') : false;
            $body_font_variant = cs_get_option('body_font_variant') ? cs_get_option('body_font_variant') : false;
            $body_font['family'] = (isset($body_font['font-family']) && !empty($body_font['font-family'])) ? $body_font['font-family'] : 'Roboto';
            $body_font['font'] = (isset($body_font['type']) && !empty($body_font['type'])) ? $body_font['type'] : 'google';
            $body_font_variant = !empty($body_font_variant) ? $body_font_variant : array(400, 700, 600, 500);
            $google_fonts = [];

            if (!empty($body_font_variant)) {
                foreach ($body_font_variant as $variant) {
                    $google_fonts[] = array(
                        'family' => $body_font['family'],
                        'variant' => $variant,
                        'font' => $body_font['font']
                    );
                }
            }
            //heading font enqueue
            $heading_font_enable = false;
            if (null == cs_get_option('heading_font_enable')) {
                $heading_font_enable = false;
            } elseif ('0' == cs_get_option('heading_font_enable')) {
                $heading_font_enable = true;
            } elseif ('1' == cs_get_option('heading_font_enable')) {
                $heading_font_enable = false;
            }
            $heading_font = cs_get_option('heading_font') ? cs_get_option('heading_font') : false;
            $heading_font_variant = cs_get_option('heading_font_variant') ? cs_get_option('heading_font_variant') : false;
            $heading_font['family'] = (isset($heading_font['font-family']) && !empty($heading_font['font-family'])) ? $heading_font['font-family'] : 'Barlow Semi Condensed';
            $heading_font['font'] = (isset($heading_font['type']) && !empty($heading_font['type'])) ? $heading_font['type'] : 'google';
            $heading_font_variant = !empty($heading_font_variant) ? $heading_font_variant : array(400, 500, 600, 700, 800);
            if (!empty($heading_font_variant) && !$heading_font_enable) {
                foreach ($heading_font_variant as $variant) {
                    $google_fonts[] = array(
                        'family' => $heading_font['family'],
                        'variant' => $variant,
                        'font' => $heading_font['font']
                    );
                }
            }

            if (!empty($google_fonts)) {
                foreach ($google_fonts as $font) {
                    if (!empty($font['font']) && $font['font'] == 'google') {
                        $variant = (!empty($font['variant']) && $font['variant'] !== 'regular') ? ':' . $font['variant'] : '';
                        if (!empty($font['family'])) {
                            $enqueue_fonts[] = $font['family'] . $variant;
                        }
                    }
                }
            }

            $enqueue_fonts = array_unique($enqueue_fonts);
            return $enqueue_fonts;
        }

        /**
         * Load Theme Css
         * @since 1.0.0
         */
        public function load_theme_css()
        {
            $theme_version = SENATORY_DEV ? time() : senatory()->get_theme_info('version');
            $css_ext = '.css';
            //load google fonts
            $enqueue_google_fonts = self::load_google_fonts();
            if (!empty($enqueue_google_fonts)) {
                wp_enqueue_style('senatory-google-fonts', esc_url(add_query_arg('family', urlencode(implode('|', $enqueue_google_fonts)), '//fonts.googleapis.com/css')), [], null);
            }
            $all_css_files = array(
                [
                    'handle' => 'animate',
                    'src' => SENATORY_CSS . '/animate.css',
                    'deps' => [],
                    'ver' => $theme_version,
                    'media' => 'all',
                ],

                [
                    'handle' => 'icomoonfont',
                    'src' => SENATORY_CSS . '/icomoon.css',
                    'deps' => [],
                    'ver' => $theme_version,
                    'media' => 'all',
                ],

                [
                    'handle' => 'bootstrap',
                    'src' => SENATORY_CSS . '/bootstrap.min.css',
                    'deps' => [],
                    'ver' => $theme_version,
                    'media' => 'all',
                ],

                [
                    'handle' => 'hover-min',
                    'src' => SENATORY_CSS . '/hover-min.css',
                    'deps' => [],
                    'ver' => $theme_version,
                    'media' => 'all',
                ],

                [
                    'handle' => 'font-awesome',
                    'src' => SENATORY_CSS . '/font-awesome.min.css',
                    'deps' => [],
                    'ver' => '5.12.0',
                    'media' => 'all',
                ],

                [
                    'handle' => 'magnific-popup',
                    'src' => SENATORY_CSS . '/magnific-popup.css',
                    'deps' => [],
                    'ver' => $theme_version,
                    'media' => 'all',
                ],

                [
                    'handle' => 'owl-carousel',
                    'src' => SENATORY_CSS . '/owl.carousel.min.css',
                    'deps' => [],
                    'ver' => $theme_version,
                    'media' => 'all',
                ],

                [
                    'handle' => 'slick',
                    'src' => SENATORY_CSS . '/slick.css',
                    'deps' => [],
                    'ver' => $theme_version,
                    'media' => 'all',
                ],

                [
                    'handle' => 'nice',
                    'src' => SENATORY_CSS . '/nice-select.css',
                    'deps' => [],
                    'ver' => $theme_version,
                    'media' => 'all',
                ],

                [
                    'handle' => 'senatory-main-style',
                    'src' => SENATORY_CSS . '/main-style' . $css_ext,
                    'deps' => [],
                    'ver' => $theme_version,
                    'media' => 'all',
                ],

                [
                    'handle' => 'senatory-responsive',
                    'src' => SENATORY_CSS . '/responsive' . $css_ext,
                    'deps' => [],
                    'ver' => $theme_version,
                    'media' => 'all',
                ],
            );

            if (class_exists('WooCommerce')) {
                $all_css_files[] = array(
                    'handle' => 'senatory-woocommerce-style',
                    'src' => SENATORY_CSS . '/woocommerce-style' . $css_ext,
                    'deps' => array(),
                    'ver' => $theme_version,
                    'media' => 'all',
                );
            }
            $all_css_files = apply_filters('senatory_theme_enqueue_style', $all_css_files);

            if (is_array($all_css_files) && !empty($all_css_files)) {
                foreach ($all_css_files as $css) {
                    call_user_func_array('wp_enqueue_style', $css);
                }
            }
            wp_enqueue_style('senatory-style', get_stylesheet_uri());

            if (senatory()->is_senatory_core_active()) {
                if (file_exists(SENATORY_DYNAMIC_STYLESHEETS . '/theme-inline-css-style.php')) {
                    require_once SENATORY_DYNAMIC_STYLESHEETS . '/theme-inline-css-style.php';
                    require_once SENATORY_DYNAMIC_STYLESHEETS . '/theme-option-css-style.php';
                    wp_add_inline_style('senatory-style', senatory()->minify_css_lines($GLOBALS['senatory_inline_css']));
                    wp_add_inline_style('senatory-style', senatory()->minify_css_lines($GLOBALS['theme_customize_css']));
                }
            }
        }

        /**
         * Load Theme js
         * @since 1.0.0
         */
        public function load_theme_js()
        {
            $theme_version = senatory()->get_theme_info('version');
            $js_ext = SENATORY_DEV ? '.js' : '.min.js';
            
            $scripts = [
            ['handle' => 'bootstrap', 'src'=> SENATORY_JS . '/bootstrap.min.js','dep'=> array( 'jquery' ),'version'=> $theme_version,'in_foot'=> true],
            ['handle' => 'popper-js', 'src'=> SENATORY_JS . '/popper.min.js', 'dep'=> array( 'jquery' ),'version'=> $theme_version,'in_foot'=> true],
            ['handle' => 'magnific-popup', 'src'=> SENATORY_JS . '/jquery.magnific-popup.js', 'dep'=> array( 'jquery' ),'version'=> $theme_version,'in_foot'=> true],
            ['handle' => 'wow-js', 'src'=> SENATORY_JS . '/wow.min.js', 'dep'=> array( 'jquery' ),'version'=> $theme_version,'in_foot'=> true],
            ['handle' => 'owl-carousel', 'src'=> SENATORY_JS . '/owl.carousel.min.js', 'dep'=> array( 'jquery' ),'version'=> $theme_version,'in_foot'=> true],
            ['handle' => 'waypoints-js', 'src'=> SENATORY_JS . '/waypoints.min.js', 'dep'=> array( 'jquery' ),'version'=> $theme_version,'in_foot'=> true],
            ['handle' => 'counterup-js', 'src'=> SENATORY_JS . '/jquery.counterup.min.js', 'dep'=> array( 'jquery' ),'version'=> $theme_version,'in_foot'=> true],
            ['handle' => 'slick-js', 'src'=> SENATORY_JS . '/slick.min.js', 'dep'=> array( 'jquery' ),'version'=> $theme_version,'in_foot'=> true],
            ['handle' => 'niceselect-js', 'src'=> SENATORY_JS . '/jquery.nice-select.js', 'dep'=> array( 'jquery' ),'version'=> $theme_version,'in_foot'=> true],
            ['handle' => 'rprogressbar', 'src'=> SENATORY_JS . '/jQuery.rProgressbar' . $js_ext, 'dep'=> array( 'jquery' ),'version'=> $theme_version,'in_foot'=> true],
            ['handle' => 'senatory-main-script', 'src'=> SENATORY_JS . '/main' . $js_ext, 'dep'=> array( 'jquery' ),'version'=> $theme_version,'in_foot'=> true]
            ];
			
            for ($i=0; $i < sizeof($scripts); $i++) {
                wp_enqueue_script( $scripts[$i]['handle'], $scripts[$i]['src'], $scripts[$i]['dep'], $scripts[$i]['version'], $scripts[$i]['in_foot'] );    
            }

            if (is_singular() && comments_open() && get_option('thread_comments')) {
                wp_enqueue_script('comment-reply');
            }
        }

        /**
         * Load THeme Dependency Files
         * @since 1.0.0
         */
        public function load_theme_dependency_files()
        {
            $includes_files = array(

                array(
                    'file-name' => 'activation',
                    'file-path' => SENATORY_TGMA
                ),

                array(
                    'file-name' => 'theme-breadcrumb',
                    'file-path' => SENATORY_INC
                ),
                array(
                    'file-name' => 'theme-excerpt',
                    'file-path' => SENATORY_INC
                ),
                array(
                    'file-name' => 'theme-hook-customize',
                    'file-path' => SENATORY_INC
                ),
                array(
                    'file-name' => 'theme-comments-modifications',
                    'file-path' => SENATORY_INC
                ),
                array(
                    'file-name' => 'customizer',
                    'file-path' => SENATORY_INC
                ),
                array(
                    'file-name' => 'theme-megamenu-walker',
                    'file-path' => SENATORY_INC
                ),
                array(
                    'file-name' => 'theme-group-fields-cs',
                    'file-path' => SENATORY_THEME_SETTINGS
                ),
                array(
                    'file-name' => 'theme-group-fields-value-cs',
                    'file-path' => SENATORY_THEME_SETTINGS
                ),
                array(
                    'file-name' => 'theme-metabox-cs',
                    'file-path' => SENATORY_THEME_SETTINGS
                ),

                array(
                    'file-name' => 'theme-shortcode-option-cs',
                    'file-path' => SENATORY_THEME_SETTINGS
                ),
                array(
                    'file-name' => 'theme-customizer-cs',
                    'file-path' => SENATORY_THEME_SETTINGS
                ),
                array(
                    'file-name' => 'theme-option-cs',
                    'file-path' => SENATORY_THEME_SETTINGS
                ),
            );

            if (class_exists('WooCommerce')) {
                $includes_files[] = array(
                    'file-name' => 'theme-woocommerce-customize',
                    'file-path' => SENATORY_INC
                );
            }
            if (is_array($includes_files) && !empty($includes_files)) {
                foreach ($includes_files as $file) {
                    if (file_exists($file['file-path'] . '/' . $file['file-name'] . '.php')) {
                        require_once $file['file-path'] . '/' . $file['file-name'] . '.php';
                    }
                }
            }

        }

        /**
         * Add editor style
         * @since 1.0.0
         */
        public function add_editor_styles()
        {
            add_editor_style(get_template_directory_uri() . '/assets/css/editor-style.css');
        }
        /**
         * Add gutenberg editor script 
         * @hook enqueue_block_editor_assets
         */

        public function gutenbarg_script(){
            
            $enqueue_google_fonts = self::load_google_fonts();
            if (!empty($enqueue_google_fonts)) {
                wp_enqueue_style('senatory-google-fonts', esc_url(add_query_arg('family', urlencode(implode('|', $enqueue_google_fonts)), '//fonts.googleapis.com/css')), [], null);
            }
            $theme_version = SENATORY_DEV ? time() : senatory()->get_theme_info('version');
            wp_enqueue_style('icomoonfont-editor', SENATORY_CSS.'/icomoon.css', [], $theme_version, 'all'); 
            wp_enqueue_style('flynext-gutenbarg', SENATORY_CSS.'/gutenbarg.css', [], $theme_version, 'all'); 
        }

    }//end class
    if (class_exists('senatory_Init')) {
        senatory_Init::getInstance();
    }
}